<?php

namespace App\Http\Controllers\Api;

use App\Enums\HttpCodesEnum;
use App\Enums\PaymentsEnums;
use App\Enums\SadadStatusEnums;
use App\Http\Controllers\Controller;
use App\Http\Requests\CategoryPackagesRequest;
use App\Http\Requests\initialSubscriptionRequest;
use App\Http\Requests\paymentStatusRequest;
use App\Http\Resources\CategoryPackageResource;
use App\Http\Resources\CategoryResource;
use App\Http\Resources\SubscriptionsResource;
use App\Models\Category;
use App\Models\Marketer;
use App\Models\Package;
use App\Models\PackageHistory;
use App\Models\PaymentTransaction;
use App\Models\Setting;
use App\Models\ShopCategory;
use App\Models\User;
use Applab\Sadad\Facades\Sadad;
use Applab\Sadad\Requests\WCORequest;
use Carbon\Carbon;
use Illuminate\Http\Request;
use Illuminate\Support\Arr;
use Illuminate\Support\Facades\Http;
use DB;
use Str;
use Illuminate\Support\Facades\App;

class PaymentsController extends Controller
{
    public function categoryPackages(CategoryPackagesRequest $request)
    {
        $category = Category::query()
            ->where('id', $request->input('category_id'))
            ->first();

        $packages = Package::query()->where('is_active', 1)->get();

        return response()->json([
            'category' => new CategoryResource($category),
            'packages' => CategoryPackageResource::collection($packages)
        ]);
    }

    public function initialSubscription(initialSubscriptionRequest $request)
    {

        $package = Package::query()->find($request->input('package_id'));

        $free_month = (boolean) Setting::query()->where('key', 'FreeMonth')->first()->value;
        $free_package_id = (boolean) Setting::query()->where('key', 'FreePackageId')->first()->value;

        $marketer = Marketer::where('id', $request->marketer_id)->first();
        if ($marketer && $marketer->discount_percentage >= 0) {
            $discountAmount = ($package->cost) * ($marketer->discount_percentage) / 100;
            $payableAmount = $package->cost - $discountAmount;
            $marketerDetails = [];
            $marketerDetails['discount_percentage'] = $marketer->discount_percentage;
            $marketerDetails['discount_amount'] = $discountAmount;
            $marketerDetails['marketer_code'] = $marketer->marketer_code;

        } else {
            $payableAmount= $package->cost;
        }

        $payment = PaymentTransaction::query()->create([
            'user_id' => $request->user()->id,
            'category_id' => $request->input('category_id'),
            'package_id' => $request->input('package_id'),
            'amount' =>$payableAmount,
            'marketer_id' => @$marketer->id,
            'marketer_details' => @json_encode($marketerDetails),
            'payment_status' => PaymentsEnums::PENDING,
            'subscribe_days' => $package->days
        ]);
        $pay_id = HashIdObj()->encode($payment->id);
        $free_month_available = $package->id == $free_package_id ;
        $accept_more_free_month=$request->user()->categories()->where('category_id', $request->input('category_id'))->count();
        return response()->json([
            'url' => route('pay-now.show', [
                'lang' => $request->header('Accept-Language'),
                'pay_id' => $pay_id,
            ]),
            'free_month_used' => $free_month_available && ($accept_more_free_month < 1) ? false : true,
            'payment_id' => $pay_id,
            'free_month' => $free_month ? [
                'free_month_available' => $free_month_available,
                'free_month_url' => $free_month_available && $accept_more_free_month < 1 ? url('api/free-month', $pay_id) : null,
                'free_month_used_message' => $free_month_available && $accept_more_free_month < 1 ? null : trans('global.free_month_used'),
            ] : []
        ]);
    }

    public function showPaymentPage($lang, $pay_id)
    {
        App::setlocale($lang);

        $payment_id = HashIdObj()->decode($pay_id);
        $payment = PaymentTransaction::with('package', 'user')->where('id', $payment_id)->first();
        if ($payment) {
            if ($payment->status == SadadStatusEnums::SUCCESS) {//todo change to view instead of json
                return view('sadad::web-checkout-success', compact('payment'));
            } else {
                $webCheckoutReq = new WCORequest();
                $webCheckoutReq->total_amount = $payment->amount;
                $webCheckoutReq->order_id = HashIdObj()->encode($payment_id);
                $webCheckoutReq->customer_mobile = $payment->user->phone ?? $payment->user->whatsapp;
                $webCheckoutReq->callback_url = url('sadad-purchased/' . $webCheckoutReq->order_id);
                $products[] = ['title' => $payment->package->en_name, 'id' => $payment->package_id, 'quantity' => 1, 'amount' => $payment->amount, 'type' => 'line_item'];
                $webCheckoutReq->setProducts($products);
                return Sadad::webCheckoutOne($webCheckoutReq);
            }
        } else {
            return view('sadad::web-checkout-error')->withErrors(['error' => 'Invalid payment id!']);
        }
    }

    public function paymentStatus(Request $response, $order_id)
    {
        if (auth()->user()) {
            App::setLocale(auth()->user()->lang);
        }

        try {
            $payment_id = HashIdObj()->decode($order_id);
            $payment = PaymentTransaction::with('package')->where('id', $payment_id)->first();
            if ($payment) {
                if ($payment->status == SadadStatusEnums::SUCCESS) {
                    return view('sadad::web-checkout-success')->with($payment);
                } else {
                    DB::beginTransaction();
                    $payment->update([
                        'payment_id' => $response->get('ORDERID'),
                        'transaction_id' => $response->get('transaction_number'),
                        'status' => SadadStatusEnums::getByCode($response->get('RESPCODE')),
                        'amount' => $response->get('TXNAMOUNT'),
                        'response' => $response,
                    ]);

                    if ($payment->status == SadadStatusEnums::SUCCESS) {

                        $shopCategory = ShopCategory::query()
                            ->where('user_id', $payment->user_id)
                            ->where('category_id', $payment->category_id)
                            ->where('is_active', 1)
                            ->first();

                        if ($shopCategory) {
                            $validTo = Carbon::now()->addDays($payment->package->days)->addDays($shopCategory->days_left);

                            $shopCategory->update([
                                'category_id' => $payment->category_id,
                                'package_id' => $payment->package_id,
                                'valid_from' => Carbon::now(),
                                'valid_to' => $validTo,
                                'payment_id' => $payment->id,
                                'is_active' => 1,
                                'current_status_json' => json_encode($payment->package)
                            ]);

                        } else {
                            $validTo = Carbon::now()->addDays($payment->package->days);

                            $shopCategory = ShopCategory::query()->create([
                                'user_id' => $payment->user_id,
                                'category_id' => $payment->category_id,
                                'package_id' => $payment->package_id,
                                'valid_from' => Carbon::now(),
                                'valid_to' => $validTo,
                                'payment_id' => $payment->id,
                                'is_active' => 1,
                                'current_status_json' => json_encode($payment->package)
                            ]);

                        }

                        PackageHistory::create([
                            'shop_category_id' => $shopCategory->id,
                            'package_id' => $payment->package->id,
                            'from' => Carbon::now(),
                            'to' => $validTo,
                        ]);

                        Setting::query()->where('key', 'TotalPayment')->increment('value', $payment->amount);
                        DB::commit();
                        return view('sadad::web-checkout-success', compact('payment'));
                    }
                    return view('sadad::web-checkout-error')->withErrors(['error' => $response->get('RESPMSG')]);
                }
            }
        } catch (\Exception $e) {
            DB::rollback();
            throw $e;
        }
    }

    public function freeMonth($pay_id)
    {
        $payment_id = HashIdObj()->decode($pay_id);
        $payment = PaymentTransaction::with('package')->where('id', $payment_id)->first();

        if ($payment) {
            $payment->update([
                'payment_id' => $payment_id[0],
                'transaction_id' => $pay_id . '-' . Str::random(5),
                'status' => SadadStatusEnums::SUCCESS,
                'amount' => 0,
                'subscribe_days' => 30,
                'response' => 'Free Month Subscription',
            ]);

            $validTo = Carbon::now()->addDays(30);

            $shopCategory = ShopCategory::query()->create([
                'user_id' => $payment->user_id,
                'category_id' => $payment->category_id,
                'package_id' => $payment->package_id,
                'valid_from' => Carbon::now(),
                'valid_to' => $validTo,
                'payment_id' => $payment->id,
                'is_active' => 1,
                'current_status_json' => json_encode($payment->package)
            ]);

            PackageHistory::create([
                'shop_category_id' => $shopCategory->id,
                'package_id' => $payment->package->id,
                'from' => Carbon::now(),
                'to' => $validTo,
            ]);

            $status = true;
            return view('dashboard.master-data.settings.partials.free-month', compact('status'));
        } else {
            $status = false;
            return view('dashboard.master-data.settings.partials.free-month', compact('status'));
        }
    }

    public function getMySubscriptions()
    {
        $shopCategories = ShopCategory::query()
            ->where('user_id', \request()->user()->id)
            ->where('is_active', 1)
            ->get();

        return response()->json(['subscriptions' => SubscriptionsResource::collection($shopCategories)]);
    }

    function CheckResponseStatus($result)
    {
        $code = HttpCodesEnum::ValidationError;
        switch ($result) {
            case -1:
                $msg = "Unauthorized -- API key is invaild(not Valid GUID), or incomplete merchant profile.";
                break;
            case -2:
                $msg = "Not Found -- The specified orderId could not be found.";
                break;
            case -3:
                $msg = "Not Support -- merchnat application doesn't support the sent currency code.";
                break;
            case -7:
                $msg = "Not Found -- there isn't any recurring or autosave payment in Fatora gateway for this merchant with the request parameter.";
                break;
            case -8:
                $msg = "Invalid -- Token is not valid guid.";
                break;
            case -10:
                $msg = "Bad Request -- required parameters requested aren't sent in request.";
                break;
            case -20:
                $msg = "Not Found -- There isn't application data for the merchant for the sent currency in payment gateway.";
                break;
            case -21:
                $msg = "Not Support -- payment gateway doesn't support void payment";
                break;
            default:
                $msg = $result;
                $code = HttpCodesEnum::Success;
                break;
        }

        return ['code' => $code, 'message' => $msg];
    }


    public function checkMarketCode(Request $request)
    {

        $marketer = Marketer::where('marketer_code', $request->marketer_code)->where('is_active', 1)->first();
        $package = Package::query()->find($request->input('package_id'));
        if ($marketer) {
            $discountAmount = ($package->cost) * ($marketer->discount_percentage) / 100;
            $payableAmount = $package->cost - $discountAmount;
            return response()->json([
                'status' => true,
                'discount_amount' => $discountAmount,
                'payable_amount' => $payableAmount,
                'marketer_id' => $marketer->id,
                'message' => trans('global.code_is_valid')
            ]);
        } else {
            return response()->json([
                'status' => false,
                'message' => trans('global.code_invalid')
            ]);

        }

    }
}
