<?php

namespace App\Http\Controllers\Dashboard;

use App\Http\Controllers\Controller;
use App\Http\Filters\FilterModels\AdminFilter;
use App\Http\Requests\StoreAdminRequest;
use App\Http\Requests\UpdateAdminRequest;
use App\Http\Traits\UploadHandler;
use App\Models\Admin;
use App\Models\User;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Spatie\Permission\Models\Role;

class AdminsController extends Controller
{
    use UploadHandler;

    public function __construct()
    {
        $this->middleware('can:admins_index');
    }

    public function index() {
        $admins =   Admin::query()
            ->filter(new AdminFilter(\request()))
            ->paginate(10);

        return view('dashboard.administrator.admins.index', ['admins'   =>  $admins]);
    }

    public function create() {
        $roles = Role::query()->get();
        return view('dashboard.administrator.admins.create', ['roles' => $roles]);
    }

    public function store(StoreAdminRequest $request) {
        $admin = Admin::query()->create([
                'name'  =>  $request->input('name'),
                'phone'  =>  $request->input('phone'),
                'password'  =>  bcrypt($request->input('password')),
                'is_active' =>  $request->input('is_active', 0)
            ]
        );

        $admin->assignRole(Role::findById($request->input('role_id')));

        $this->uploadAdminImage($request, $admin);

        return redirect()->action('Dashboard\AdminsController@index')->with('success', trans('global.created_success'));
    }

    public function edit($id) {
        $admin   =   Admin::query()->with('roles')->findOrFail(HashIdObj()->decode($id)[0]);
        $admin_role_id = optional($admin->roles->first())->id;
        $roles = Role::query()->get();
        return view('dashboard.administrator.admins.edit', ['admin' => $admin, 'roles' => $roles, 'current_role'=>$admin_role_id]);
    }

    public function update(UpdateAdminRequest $request, $id) {
        $admin = Admin::query()->findOrFail($id);

        if (Auth::user()->hasRole(['super_admin'])) {
            $admin->update([
                    'name' => $request->input('name'),
                    'phone' => $request->input('phone'),
                    'is_active' => $admin->id != 1 ? $request->input('is_active') : 1,
                ] +
                ($request->filled('password') ? ['password' => bcrypt($request->input('password'))] : [])
            );
        }
        else{
            $admin->update([
                'name' => $request->input('name'),
                'phone' => $request->input('phone'),
            ] +
            ($request->filled('password') ? ['password' => bcrypt($request->input('password'))] : [])
        );
        }

        $this->uploadAdminImage($request, $admin);

        if (Auth::user()->hasRole(['super_admin'])) {
            $admin->syncRoles([Role::findById($request->input('role_id'))->name]);
        }

        return redirect()->action('Dashboard\AdminsController@index')->with('success', trans('global.updated_success'));
    }


    public function toggleAdminStatus(Request $request) {
        $user   =   Admin::query()->where('id', '!=' ,1)->findOrFail($request->input('id'));
        $user->update(['is_active'=> !$user->is_active]);

        return response()->json(['status'=>true,'is_active'=>$user->is_active],200);
    }

    public function destroy($id) {
        $admin   =   Admin::query()->where('id', '!=' ,1)->find($id);
        if(!$admin) {
            return redirect()->action('Dashboard\AdminsController@index')->with('error', trans('global.cannot_find_admin_user'));
        } else {
            $admin->delete();
            return redirect()->action('Dashboard\AdminsController@index')->with('success', trans('global.deleted_successful'));
        }
    }
}
