<?php

namespace App\Http\Controllers\Dashboard;

use App\Exports\PaymentTransactionsExport;
use App\Http\Controllers\Controller;
use App\Http\Filters\FilterModels\MarketersFilter;
use App\Http\Requests\MarketerRequest;
use App\Models\Marketer;
use App\Models\PaymentTransaction;
use App\Models\Product;
use App\Models\Role;
use Carbon\Carbon;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Maatwebsite\Excel\Excel;
use Psy\Util\Str;

class MarketerController extends Controller
{
    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function index()
    {
        $marketers = Marketer::query()
            ->filter(new MarketersFilter(\request()))
            ->latest()
            ->paginate();

        return view('dashboard.marketer.index', [
            'marketers' => $marketers,
        ]);
    }

    /**
     * Show the form for creating a new resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function create()
    {
        //
        return view('dashboard.marketer.create');
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param \Illuminate\Http\Request $request
     * @return \Illuminate\Http\Response
     */
    public function store(MarketerRequest $request)
    {
        DB::beginTransaction();
        try {
            if ($request->input('discount_percentage') == null) {
                $discountPercentage = 0;
            } else {
                $discountPercentage = $request->input('discount_percentage');
            }
            $product = Marketer::query()->create([
                    'name_ar' => $request->input('name_ar'),
                    'name_en' => $request->input('name_en'),
                    'discount_percentage' => $discountPercentage,
                    'marketer_code' => $request->input('marketer_code'),
                ]
            );
            DB::commit();

            return redirect()->route('marketer.index')->with('success', trans('global.created_success'));

        } catch (\Exception $e) {
            // dd($e->getMessage());
            DB::rollBack();
            return redirect()->back()->with('error', trans('global.something_happened'));
        }
    }

    /**
     * Display the specified resource.
     *
     * @param int $id
     * @return \Illuminate\Http\Response
     */
    public function show($id)
    {
        $marketer = Marketer::query()->find($id);

        $paymentTransaction = PaymentTransaction::query()
            ->where('marketer_id', $id)
            ->whereNotNull('response')
            ->where('status', '=', 'success')
            ->whereHas('user.activeCategories')
            ->with('packagehistory')
            ->paginate();
        //dd($paymentTransaction);

        return view('dashboard.marketer.show', ['marketer' => $marketer, 'transactions' => $paymentTransaction]);
    }

    /**
     * Show the form for editing the specified resource.
     *
     * @param int $id
     * @return \Illuminate\Http\Response
     */
    public function edit($id)
    {
        $marketer = Marketer::query()->find($id);
        return view('dashboard.marketer.edit', ['marketer' => $marketer]);
    }

    /**
     * Update the specified resource in storage.
     *
     * @param \Illuminate\Http\Request $request
     * @param int $id
     * @return \Illuminate\Http\Response
     */
    public function update(MarketerRequest $request, $id)
    {
        DB::beginTransaction();
        try {
            $marketer = Marketer::query()->find($id);
            if ($marketer) {
                $marketer->update([
                        'name_ar' => $request->input('name_ar'),
                        'name_en' => $request->input('name_en'),
                        'discount_percentage' => $request->input('discount_percentage'),
                        'marketer_code' => $request->input('marketer_code'),

                    ]
                );
                DB::commit();
            }
            return redirect()->back()->with('success', trans('global.updated_success'));

        } catch (\Exception $e) {
            DB::rollBack();
            // dd($e->getMessage());
            return redirect()->back()->with('error', trans('global.something_happened'));
        }
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param int $id
     * @return \Illuminate\Http\Response
     */
    public function destroy(Marketer $marketer)
    {

        $marketer->delete();
        return redirect()->route('marketer.index')->with('success', trans('global.deleted_successful'));
    }

    public function export(Request $request, $marketer_id)
    {
        $query = PaymentTransaction::query()
            ->with(['user' => function ($q) {
                $q->withTrashed();
            }])
            ->where(function ($query) use ($marketer_id) {
                $query->where('marketer_id', $marketer_id);
            })
            ->whereNotNull('response')
            ->where('status', '=', 'success')
            ->whereHas('user.activeCategories')
            ->with('packagehistory');
        if ($request->sub_date != null) {
            $query = $query->whereDate('created_at', Carbon::parse($request->sub_date)->format('Y-m-d'));
        }
        $payment = $query->get();
        return \Maatwebsite\Excel\Facades\Excel::download(new PaymentTransactionsExport($payment), 'subscribers-' . date('dmYhia') . '.xlsx');
    }

    public function getMarketerCode(Request $request)
    {

        $marketerCode = Marketer::latest()->first();
        return response()->json(['status' => true, 'marketerCode' => $marketerCode->id + 1], 200);

    }

    public function searchSubscriber(Request $request)
    {
        $marketer = Marketer::query()->find($request->id);
        $date = Carbon::parse($request->searchKey)->format('Y-m-d');
        $transactions = PaymentTransaction::query()
            ->where('marketer_id', $request->id)
            ->whereNotNull('response')
            ->where('status', '=', 'success')
            ->whereHas('user.activeCategories')
            ->with('packagehistory')
            ->whereDate('created_at', $date)
            ->paginate();
        return response()->json([
            'status' => 'success',
            'html' => view('dashboard.marketer.subscribers', compact('transactions', 'marketer'))->render()
        ]);

    }

}
