<?php

namespace App\Http\Controllers\Dashboard;

use App\Enums\NotificationTypesEnum;
use App\Enums\UsersTypesEnums;
use App\Http\Controllers\Controller;
use App\Http\Filters\FilterModels\ProductsFilter;
use App\Http\Requests\AdminCreateProductRequest;
use App\Http\Requests\AdminUpdateProductRequest;
use App\Http\Requests\UpdateProductRequest;
use App\Http\Traits\UploadHandler;
use App\Models\Category;
use App\Models\FirebaseNotification;
use App\Models\Product;
use App\Models\User;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;

class ProductsController extends Controller
{
    use UploadHandler;

    public function index()
    {
        $products = Product::query()
            ->filter(new ProductsFilter(\request()))
            ->with('shop', 'category')
            ->latest()
            ->paginate();

        $categories = Category::query()->select('id', 'ar_name', 'en_name')->get();

        $shops = User::query()->where('user_type', UsersTypesEnums::Vendor)->get();

        return view('dashboard.users-management.products.index', [
            'products' => $products,
            'categories' => $categories,
            'shops' => $shops
        ]);
    }

    public function create()
    {

        $shops = User::query()->where('user_type', UsersTypesEnums::Vendor)->get();
        $categories = Category::query()->get();
        return view('dashboard.users-management.products.create', ['categories' => $categories, 'shops' => $shops]);
    }

    public function edit($id)
    {

        $product = Product::query()->find($id);

        $categories = Category::query()->select('id', 'ar_name', 'en_name')->get();

        return view('dashboard.users-management.products.edit', ['categories' => $categories, 'product' => $product]);
    }

    public function store(AdminCreateProductRequest $request)
    {
        DB::beginTransaction();
        try {
            $product = Product::query()->create([
                    'name_ar' => $request->input('name_ar'),
                    'name_en' => $request->input('name_en'),
                    'ar_description' => $request->input('ar_description'),
                    'en_description' => $request->input('en_description'),
                    'category_id' => $request->input('category_id'),
                    'price' => $request->input('price'),
                    // 'price_after_discount'  =>  $request->input('price_after_discount'),
                    'user_id' => $request->input('user_id'),
                    'is_active' => $request->input('is_active')
                ]
            );
            $this->storeProductImages($request, $product);
            DB::commit();

            return redirect()->route('products.index')->with('success', trans('global.created_success'));

        } catch (\Exception $e) {
            // dd($e->getMessage());
            DB::rollBack();
            return redirect()->back()->with('error', trans('global.something_happened'));
        }
    }

    public function update(AdminUpdateProductRequest $request, $id)
    {
        DB::beginTransaction();
        try {
            $product = Product::query()->find($id);
            if ($product) {
                $product->update([
                        'name_ar' => $request->input('name_ar'),
                        'name_en' => $request->input('name_en'),
                        'ar_description' => $request->input('ar_description'),
                        'en_description' => $request->input('en_description'),
                        'category_id' => $request->input('category_id'),
                        'price' => $request->input('price'),
                        // 'price_after_discount'  =>  $request->input('price_after_discount'),
                        'is_active' => $request->input('is_active')
                    ]
                );

                // $this->deleteProductImages($request, $product);
                $this->storeProductImages($request, $product);
                DB::commit();
            } else {
                throw new \Exception(trans('global.cannot_find_product'));
            }

            return redirect()->back()->with('success', trans('global.updated_success'));

        } catch (\Exception $e) {
            DB::rollBack();
            return redirect()->back()->with('error', trans('global.something_happened'));
        }
    }

    public function toggleProductStatus(Request $request)
    {
        $product = Product::query()->find($request->input('id'));
        $product->update(['is_active' => !$product->is_active]);
        return response()->json(['status' => true, 'is_active' => $product->is_active], 200);
    }

    public function toggleProductApproval(Request $request)
    {
        $product = Product::query()->find($request->input('id'));
        $product->update(['is_approved' => 1]);

        $path = optional(optional(optional($product->images)->filter(function ($image) {
            return $image->is_default;
        }))->first())->full_path;
        $title_en = trans('global.product_approved_notification_title', [], 'en');
        $body_en = trans('global.product_approved_notification_body', [
            'product_name' => $product->id,
        ], 'en');

        FirebaseNotification::query()->create([
            'user_id' => $product->user_id,
            'title' => $title_en,
            'body' => $body_en,
            'lang' => 'en',
            'image' => $path,
            'type' => NotificationTypesEnum::Product
        ]);

        $title_ar = trans('global.product_approved_notification_title', [], 'ar');
        $body_ar = trans('global.product_approved_notification_body', [
            'product_name' => $product->id,
        ], 'ar');

        FirebaseNotification::query()->create([
            'user_id' => $product->user_id,
            'title' => $title_ar,
            'body' => $body_ar,
            'lang' => 'ar',
            'image' => $path,
            'type' => NotificationTypesEnum::Product
        ]);
        $title=$title_en."\n".$title_ar;
        $body =$body_en ."\n".$body_ar;

        $userId = HashIdObj()->encode($product->user_id);
        fcm()
            ->toTopic("user-$userId")
            ->priority('normal')
            ->timeToLive(0)
            ->notification([
                'title' => $title,
                'body' => $body,
                'image' => $path,
                'type' => NotificationTypesEnum::Product,
                'id' => HashIdObj()->encode($product->id),
            ])->data([
                'title' => $title,
                'body' => $body,
                'image' => $path,
                'type' => NotificationTypesEnum::Product,
                'id' => HashIdObj()->encode($product->id),
            ])
            ->send();

        return response()->json(['status' => true, 'is_approved' => $product->is_approved], 200);
    }

    public function disapproveProduct(Request $request)
    {

        DB::beginTransaction();
        try {
            $product = Product::query()->find($request->input('id'));
            $path = optional(optional(optional($product->images)->filter(function ($image) {
                return $image->is_default;
            }))->first())->full_path;

            $product->update([
                'is_approved' => 0,
                'declined_reason' => $request->comment,
                'declined_reason_en' => $request->comment_en,
            ]);
            $title_en = trans('global.product_declined_notification_title',[],'en');
            $body_en = trans('global.product_declined_notification_body', [
                'product_name' => $product->id,
                "comment" => $request->comment_en ?? '-',
            ],'en');
            FirebaseNotification::query()->create([
                'user_id' => $product->user_id,
                'title' => $title_en,
                'body' => $body_en,
                'lang'      =>  'en',
                'image' => $path,
                'type' => NotificationTypesEnum::Product
            ]);

            $title_ar = trans('global.product_declined_notification_title',[],'ar');
            $body_ar = trans('global.product_declined_notification_body', [
                'product_name' => $product->id,
                "comment" => $request->comment ?? '-',
            ],'ar');

            FirebaseNotification::query()->create([
                'user_id' => $product->user_id,
                'title' => $title_ar,
                'body' => $body_ar,
                'image' => $path,
                'lang'      =>  'ar',
                'type' => NotificationTypesEnum::Product
            ]);
            $title=$title_en."\n".$title_ar;
            $body =$body_en ."\n".$body_ar;

            $userId = HashIdObj()->encode($product->user_id);
            fcm()
                ->toTopic("user-$userId")
                ->priority('normal')
                ->timeToLive(0)
                ->notification([
                    'title' => $title,
                    'body' =>$body,
                    'image' => $path,
                    'type' => NotificationTypesEnum::Product,
                    'id' => HashIdObj()->encode($product->id),
                ])->data([
                    'title' =>$title,
                    'body' => $body,
                    'image' => $path,
                    'type' => NotificationTypesEnum::Product,
                    'id' => HashIdObj()->encode($product->id),
                ])
                ->send();
            DB::commit();
            return response()->json(['status' => true, 'is_approved' => $product->is_approved], 200);

        } catch (\Exception $exception) {
            DB::rollback();
            return false;
        }

    }

    public function destroy(Request $request, $id)
    {
        $product = Product::query()->find($id);
        $path = optional(optional(optional($product->images)->filter(function ($image) {
            return $image->is_default;
        }))->first())->full_path;

        if (!$product) {
            return redirect()->back()->with('error', trans('global.cannot_find_product'));
        } else {
            $title_en = trans('global.delete_message_title',[],'en');
            $body_en = trans('global.product_declined_notification_body', [
                'product_name' => $product->id,
                'comment' => $request->reason ?? '-',
            ],'en');
            FirebaseNotification::query()->create([
                'user_id' => $product->user_id,
                'title' => $title_en,
                'body' => $body_en,
                'lang' => 'en',
                'image' => $path,
                'type' => NotificationTypesEnum::Product
            ]);

            $title_ar = trans('global.delete_message_title',[],'ar');
            $body_ar= trans('global.product_declined_notification_body', [
                'product_name' => $product->id,
                'comment' => $request->reason_en ?? '-',
            ],'ar');
            FirebaseNotification::query()->create([
                'user_id' => $product->user_id,
                'title' => $title_ar,
                'body' => $body_ar,
                'lang' => 'ar',
                'image' => $path,
                'type' => NotificationTypesEnum::Product
            ]);
            $title=$title_en."\n".$title_ar;
            $body =$body_en ."\n".$body_ar;

            $userId = HashIdObj()->encode($product->user_id);
            fcm()
                ->toTopic("user-$userId")
                ->priority('normal')
                ->timeToLive(0)
                ->notification([
                    'title' =>$title,
                    'body' => $body,
                    'image' => $path,
                    'type' => NotificationTypesEnum::Product,
                    'id' => HashIdObj()->encode($product->id),
                ])
                ->data([
                    'title' => $title,
                    'body' => $body,
                    'image' => $path,
                    'type' => NotificationTypesEnum::Product,
                    'id' => HashIdObj()->encode($product->id),
                ])
                ->send();
            $product->delete();
            return redirect()->back()->with('success', trans('global.deleted_successful'));
        }
    }

    public function destroyImage(Request $request)
    {
        $product = Product::find($request->id);

        if ($product->images()->count() > 1) {
            $product->images()->where('id', $request->image_id)->first()->delete();

            return response()->json([
                'status' => true,
                'success' => 'Image deleted successfully',
            ]);
        } else {
            return response()->json([
                'status' => false,
                'success' => __('At least one image is needed'),
            ]);
        }
    }

}
