<?php

namespace App\Http\Controllers\Dashboard;

use App\Enums\UsersTypesEnums;
use App\Http\Controllers\Controller;
use App\Http\Filters\AbstractFilter;
use App\Http\Filters\FilterModels\UsersFilter;
use App\Http\Requests\StoreUserRequest;
use App\Http\Requests\UpdateUserRequest;
use App\Http\Traits\UploadHandler;
use App\Models\Country;
use App\Models\User;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;

class UsersController extends Controller
{
    use UploadHandler;

    public function index()
    {
        $users = User::query()
            ->filter(new UsersFilter(\request()))
            ->where(function ($q) {
                $q
                    ->where('user_type', UsersTypesEnums::User)
                    ->orWhere('user_vendor', true);
            })
            ->paginate();


        return view('dashboard.users-management.users.index', ['users' => $users]);
    }

    public function create()
    {
        $countries = Country::query()->get();
        return view('dashboard.users-management.users.create', ['countries' => $countries]);
    }

    public function store(StoreUserRequest $request)
    {
        DB::beginTransaction();
        try {
            $user = User::query()->create([
                    'name' => $request->input('name'),
                    'phone' => $request->input('phone'),
                    'email' => $request->input('email'),
                    'password' => bcrypt($request->input('password')),
                    'is_active' => $request->input('is_active', 0),
                    'user_type' => UsersTypesEnums::User,
                    'country_id' => $request->input('country_id'),
                ]
            );

            $this->uploadShopImage($request, $user);
            DB::commit();
            return redirect()->action('Dashboard\UsersController@index')->with('success', trans('global.created_success'));

        } catch (\Exception $e) {
            DB::rollBack();
            return redirect()->back()->with('error', trans('global.something_happened'));
        }

    }

    public function edit($id)
    {
        $user = User::query()->find($id);
        $countries = Country::query()->get();
        return view('dashboard.users-management.users.edit', ['countries' => $countries, 'user' => $user]);
    }

    public function update(UpdateUserRequest $request, $id)
    {
        DB::beginTransaction();
        try {
            $user = User::query()
                ->where(function ($q) {
                    $q->where('user_type', UsersTypesEnums::User)->orWhere('user_vendor', true);
                })
                ->find($id);
            $user->update([
                    'name' => $request->input('name'),
                    'phone' => $request->input('phone'),
                    'email' => $request->input('email'),
                    'is_active' => $request->input('is_active', 0),
                    'user_type' => UsersTypesEnums::User,
                    'country_id' => $request->input('country_id'),
                    'lang' => $request->input('lang'),
                ] +
                ($request->has('password') ? ['password' => bcrypt($request->input('password'))] : [])
            );
            if ($request->is_active == 0) {
                $user->tokens()->delete();
            }
            $this->uploadShopImage($request, $user);
            DB::commit();
            return redirect()->action('Dashboard\UsersController@index')->with('success', trans('global.updated_success'));

        } catch (\Exception $e) {
            DB::rollBack();
            return redirect()->back()->with('error', trans('global.something_happened'));
        }

    }

    public function toggleUserStatus(Request $request)
    {
        $user = User::query()
            ->where(function ($q) {
                $q->where('user_type', UsersTypesEnums::User)
                    ->orWhere('user_vendor', true);
            })
            ->find($request->input('id'));

        $user->update(['is_active' => !$user->is_active]);
        if (!$user->is_active) {
            $user->tokens()->delete();
        }
        return response()->json(['status' => true, 'is_active' => $user->is_active], 200);
    }

    public function destroy($id)
    {
        $user = User::query()
            ->where(function ($q) {
                $q->where('user_type', UsersTypesEnums::User)->orWhere('user_vendor', true);
            })
            ->find($id);
        if (!$user) {
            return redirect()->back()->with('error', trans('global.cannot_find_user'));
        } else {
            $user->update(['phone' => "$user->phone-" . now()->format('Y-m-d H:i:s')]);
            $user->tokens()->delete();
            $user->delete();
            return redirect()->back()->with('success', trans('global.deleted_successful'));
        }
    }
}
